//**************************************************************************/
// Copyright (c) 1998-2006 Autodesk, Inc.
// All rights reserved.
// 
// These coded instructions, statements, and computer programs contain
// unpublished proprietary information written by Autodesk, Inc., and are
// protected by Federal copyright law. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without
// the prior written consent of Autodesk, Inc.
//**************************************************************************/
// DESCRIPTION: Main Dx10 Effect file for GFX layer
// AUTHOR: Neil Hazzard - created February.02.2006
//***************************************************************************/


//Global Valriables
// NOTE: This has to match the D3D10SceneLight structure in CD3D10RenderEngine.h
#define NumberOfTextures 2

struct Light
{
	float4 Position;
	float4 Direction;
	float4 Diffuse;
	float4 Specular;
	float4 Ambient;
	float4 Atten;
	float4 LightData;		//x=enabled, y=type 0 omni,1 spot, 2 direct; z= falloff w= hotspot
};

struct Material
{
	float4 Diffuse;
	float4 Ambient;
	float4 Specular;
	float4 SpecularData;	//x=specamnt, y = speclevel
	float4 Emissive;
};

struct TextureInfo
{
	float4 TextureData;	//used to pass data, x=Enabled, y = sampling 0=Point 1=linear
	matrix TextureTM;
};

cbuffer cbLights
{
	Light g_Lights[8];
};

cbuffer cbMaterial
{
	Material g_Material;

};

cbuffer cbTexture
{
	TextureInfo g_Texture[NumberOfTextures];
};

cbuffer cbChangesEveryFrame
{
	matrix g_World; 
	matrix g_View ;
	matrix g_Projection ;
	matrix g_InvView;
};

cbuffer cpPerTech
{
	float4 g_Diffuse = {0.0f,0.0f,0.0f,1.0f};
	float4 g_PointColor = {1.0f,0.0f,0.0f,1.0f};
	float g_PointSize = 0.01f; 
	int g_LightCount = 8;
	bool g_UseFlatShading = false;
	bool g_UseDepthBias = false;
	bool g_EnableLighting = true;
	bool g_EnableSpecular = true;
	
};


Texture2D map[NumberOfTextures];

//! totally temp until I can get the above to work....
Texture2D opacityMap;

float Bias = 1.0e-3f;

DepthStencilState Depth
{
    DepthEnable = TRUE;
//    DepthFunc = 4;//LESSEQUAL
    DepthWriteMask = ALL;
};

RasterizerState MarkerCulling
{
	FillMode = SOLID;
	CullMode = NONE;
};

//Pipeline states

int filterType;
int MaxAddressU;
int MaxAddressV;


SamplerState samplerPoint
{
	filter =   MIN_MAG_MIP_POINT;
	AddressU = Clamp;
	AddressV = Clamp;
};

SamplerState samplerLinear
{
	filter =   MIN_MAG_MIP_LINEAR;
	AddressU = Wrap;
	AddressV = Wrap;
};

/*
SamplerState samplerTest
{
	filter =   filterType;
	AddressU = MaxAddressU;
	AddressV = MaxAddressU;
};

sampler2D samplerPoint=sampler_state
{
	texture = map[0];
	filter =   MIN_MAG_MIP_POINT;
    AddressU = CLAMP;
    AddressV = CLAMP;	
	
};

sampler2D samplerLinear=sampler_state
{
	texture = map[0];
	filter =   MIN_MAG_MIP_LINEAR;
    AddressU = WRAP;
    AddressV = WRAP;	
	
};

sampler2D samplerOpacity = sampler_state
{
	texture = map[1];
	filter =   MIN_MAG_MIP_LINEAR;
    AddressU = WRAP;
    AddressV = WRAP;	
  
};
*/

DepthStencilState DisableDepth
{
    DepthEnable = FALSE;
    DepthWriteMask = ZERO;
};

DepthStencilState EnableDepth
{
    DepthEnable = TRUE;
    DepthWriteMask = ALL;
};

DepthStencilState EnableDepthLessEqual
{
    DepthEnable = TRUE;
    DepthWriteMask = ALL;
    DepthFunc = LESS_EQUAL;
};


BlendState Blending
{
    AlphaToCoverageEnable = TRUE;
    BlendEnable[0] = TRUE;
    RenderTargetWriteMask[0] = 0xF;
};

BlendState NoBlending
{
    AlphaToCoverageEnable = FALSE;
    BlendEnable[0] = FALSE;
    RenderTargetWriteMask[0] = 0xF;
};


BlendState TextBlending
{
    AlphaToCoverageEnable = FALSE;
    BlendEnable[0] = TRUE;
    RenderTargetWriteMask[0] = 0x0F;
	
};

RasterizerState CullingCW
{
	FillMode = SOLID;
	CullMode = BACK;
	FrontCounterClockwise = TRUE;
	
	DepthBias = 0;
	DepthBiasClamp = 0.02;
	SlopeScaledDepthBias = 1.0;
};

RasterizerState DataCulling
{
	FillMode = SOLID;
	CullMode = BACK;
	FrontCounterClockwise = false;
};

//--------------------------------------------------------------------------------------
// VertexShader I/O
//--------------------------------------------------------------------------------------
struct VSPosCol
{
    float4 Position : Position;
    float4 Color	: COLOR;
    
};

struct VSPosTex0
{
    float4 Position : Position;
    float2 UV		: Texcoord;
};

struct VSPosColTex0
{
    float4 Position : Position;
    float4 Diffuse	: COLOR;
    float2 UV		: Texcoord;
};


struct VSPosNorm
{
	float4 Position : Position;
	float3 Norm		: Normal;
};

struct VSPosNormCol
{
	float4 Position : Position;
	float3 Norm		: Normal;
	float4 Color	:COLOR;
};

struct VSPosNormTex0
{
	float4 Position : Position;
	float3 Norm		: Normal;
	float2 UV		: Texcoord;
};

struct VSPosNormColTex0
{
	float4 Position : Position;
	float3 Norm		: Normal;
	float4 Color	: COLOR;
	float2 UV		: Texcoord;
};

struct PSLine
{
    float4 Position : SV_POSITION;
    float4 Diffuse : COLOR0;    
};

struct PSMesh
{
    float4 Position : SV_POSITION;
    float2 UV0		: TEXCOORD0;
    float2 UV1		: TEXCOORD1;
    float4 WorldPos	: TEXCOORD2;
    float4 Normal	: TEXCOORD3;
    float4 EyeVector: TEXCOORD4;
    float4 Diffuse : COLOR0;
    float4 Specular : COLOR1;    
};



struct GSOutput
{
    float4 Position : SV_Position;
    float3 Color : COLOR;
};

struct ColorsOutput
{
    float4 Diffuse;
    float4 Specular;
};

float4 CalcLighting( float3 worldNormal, float3 eyeVector, float3 worldPos, float4 matDiffuse, float4 matSpecular)
{
	float4 lightColor = float4(0.0,0.0,0.0,1.0);
	
    for(int i=0; i<g_LightCount; i++)
    {
		if(g_Lights[i].LightData.x == 1.0f)
		{
			float3 toLight; 
			float fAtten = 1.0;
				
			if(g_Lights[i].LightData.y == 2.0f)
				toLight = -g_Lights[i].Direction.xyz;
			else if(g_Lights[i].LightData.y == 1.0f || g_Lights[i].LightData.y == 0.0f) //spotlight,omni
			{
				toLight = normalize(g_Lights[i].Position.xyz - worldPos);
				if(g_Lights[i].LightData.y == 1.0f)
				{
					float fallOff = g_Lights[i].LightData.z ;
					float hotSpot = g_Lights[i].LightData.w  ;
					fAtten = fAtten * smoothstep(cos(fallOff),cos(hotSpot), dot(toLight,-g_Lights[i].Direction));

				}
			}
			
			float3 halfAngle = normalize( eyeVector + toLight);
			float f = clamp(dot(worldNormal,halfAngle),0,1);
			f = pow (f,g_Material.SpecularData.y);
			f = f * g_Material.SpecularData.x;
				
			//diffuse term
			lightColor = lightColor + g_Lights[i].Diffuse * matDiffuse * clamp(dot(worldNormal,toLight),0,1) * fAtten;
			
			//Spec term
			if(g_EnableSpecular)
				lightColor = lightColor + g_Lights[i].Diffuse * matSpecular * f * fAtten;
		
		}
    }
    
    return lightColor;
}



//--------------------------------------------------------------------------------------
// This shader computes standard transform and lighting
//--------------------------------------------------------------------------------------

PSLine VSG( VSPosCol input )
{
    PSLine Out = (PSLine)0;
	matrix WorldView = mul(g_World,g_View);
	Out.Position = mul(input.Position,WorldView);    	 
	Out.Position.z  -= Bias;
	Out.Position.w = 1.0f;
    Out.Diffuse = input.Color;
    return Out;
}
PSLine VS_PosCol( VSPosCol input )
{
    PSLine output = (PSLine)0;
    
    float4 worldPosition = mul(input.Position, (float4x4)g_World);
    output.Position = mul(worldPosition, mul(g_View, g_Projection));
    output.Diffuse = input.Color;
    return output;    
}
	
PSMesh VS_PosTex0( VSPosTex0 input )
{
    PSMesh output = (PSMesh)0;
    
    float4 worldPosition = mul(input.Position, (float4x4)g_World);
    output.Position = mul(worldPosition, mul(g_View, g_Projection));
    output.UV0 = input.UV;
    return output;    
}

PSMesh VS_PosColTex0( VSPosColTex0 input )
{
    PSMesh output = (PSMesh)0;
    
    float4 worldPosition = mul(input.Position, (float4x4)g_World);
    output.Position = mul(worldPosition, mul(g_View, g_Projection));
    output.UV0 = input.UV;
    output.Diffuse = input.Diffuse;
    return output;    
}


PSMesh VS_PosNorm ( VSPosNorm input)
{
	PSMesh output = (PSMesh)0;
    float4 worldPosition = mul(input.Position, (float4x4)g_World);
    float4 cameraPos = g_InvView[3];
    output.Position = mul(worldPosition, mul(g_View, g_Projection));
    if(g_UseDepthBias)
		output.Position.z -= 0.02f;
	output.Normal = mul(float4(input.Norm,0.0f),(float4x4)g_World);
    output.WorldPos = worldPosition;
    output.EyeVector = cameraPos - worldPosition;
    
	return output;

}


PSMesh VS_PosNormCol ( VSPosNormCol input)
{
	PSMesh output = (PSMesh)0;
    float4 worldPosition = mul(input.Position, (float4x4)g_World);
    float4 cameraPos = g_InvView[3];
    output.Position = mul(worldPosition, mul(g_View, g_Projection));
    if(g_UseDepthBias)
		output.Position.z -= 0.02f;
    output.EyeVector = cameraPos - worldPosition;
	output.Normal = mul(float4(input.Norm,0.0f),(float4x4)g_World);
    output.Diffuse = input.Color;
	output.WorldPos = worldPosition;    
	return output;
}
PSMesh VS_PosNormColTex0 ( VSPosNormColTex0 input)
{
	PSMesh output = (PSMesh)0;
    float4 worldPosition = mul(input.Position, (float4x4)g_World);
    float4 cameraPos = g_InvView[3];
    output.Position = mul(worldPosition, mul(g_View, g_Projection));
    if(g_UseDepthBias)
		output.Position.z -= 0.02f;    
    output.EyeVector = cameraPos - worldPosition;
	output.Normal = mul(float4(input.Norm,0.0f),(float4x4)g_World);
    output.Diffuse = input.Color;
	output.WorldPos = worldPosition;    
	output.UV0 =  input.UV;
	return output;
}

PSMesh VS_PosNormTex0 ( VSPosNormTex0 input)
{
	PSMesh output = (PSMesh)0;
    float4 worldPosition = mul(input.Position, (float4x4)g_World);
	float4 cameraPos = g_InvView[3];
    output.Position = mul(worldPosition, mul(g_View, g_Projection));
    if(g_UseDepthBias)
		output.Position.z -= 0.02f;    
    output.EyeVector = cameraPos - worldPosition;
	output.Normal = mul(float4(input.Norm,0.0f),(float4x4)g_World);
	output.WorldPos = worldPosition;        
    output.UV0 =  input.UV;
	return output;
}


//--------------------------------------------------------------------------------------
// GeometryShader I/O for Points 
//--------------------------------------------------------------------------------------

void GSPointsPerVertex( PSLine In, inout TriangleStream<PSLine> outputStream )
{
    PSLine Out;  
	Out.Diffuse = In.Diffuse;
	    
	float4 center = mul(In.Position, g_Projection);
//	float4 center = In.Position;
	center.z -= 0.03f;
//	center.w = 0.0f;

    Out.Position = float4( center.x - g_PointSize, center.y - g_PointSize, center.z, center.w );
	outputStream.Append(Out);

    Out.Position = float4( center.x + g_PointSize, center.y - g_PointSize, center.z, center.w );
	outputStream.Append(Out);

    Out.Position = float4( center.x - g_PointSize, center.y + g_PointSize, center.z, center.w );
	outputStream.Append(Out);

    Out.Position = float4( center.x + g_PointSize, center.y + g_PointSize, center.z, center.w );
	outputStream.Append(Out);

	outputStream.RestartStrip();
}

[maxvertexcount(4)]
void GSPoints( point PSLine In[1], inout TriangleStream<PSLine> outputStream )
{
	GSPointsPerVertex( In[0], outputStream );
}


//--------------------------------------------------------------------------------------
// PixelShader I/O
//--------------------------------------------------------------------------------------

float4 PS_LineShader( PSLine input) : SV_Target
{
	float4 col = input.Diffuse;
	if(col.a <0.05f)
		discard;
		
	return col;
}

float4 PS_MeshShader(PSMesh input, uniform bool alphaMask, uniform bool vertexColor) : SV_Target
{

	float4 diffuse = g_Material.Diffuse;
	float3 normal = normalize(input.Normal.xyz);
	float3 eyeVector = normalize(input.EyeVector.xyz);
	float4 result;
	
	int sampling = g_Texture[0].TextureData.y;	

	if(vertexColor)
		diffuse = input.Diffuse;
	
		
// not very dynamic !!  can be made easier later if we provide more data on the material		
		
	if(g_Texture[0].TextureData.x == 1.0f) // diffuse
	{
		
		if(sampling==0) // point
		{
			float2 newUV0 = mul(input.UV0, g_Texture[0].TextureTM); 

			diffuse = map[0].Sample( samplerPoint, newUV0 );

		}
		else //default to linear for now
		{
			float2 newUV0 = mul(input.UV0, g_Texture[0].TextureTM); 
			diffuse =  map[0].Sample( samplerLinear, newUV0);
		}
	}
	
	if(g_Texture[1].TextureData.x == 1.0f) // opacity
	{
		float2 newUV0 = mul(input.UV0, g_Texture[1].TextureTM); 
		diffuse.a =  map[1].Sample( samplerLinear, newUV0).a;
	}


	if(alphaMask){
		if(vertexColor){
			diffuse = diffuse * input.Diffuse;
		}
		else
			diffuse = diffuse * g_Material.Diffuse.a;
	}

	// copy the color in case no lighting it used
	result = diffuse;
	
	if(g_EnableLighting && !g_UseFlatShading)
	{
		 result = CalcLighting( normal, eyeVector, input.WorldPos.xyz, diffuse, g_Material.Specular);
	}	

	// add the emmissive data
	result = result + g_Material.Emissive;
	
	// re add the alpha - this will be wiped out by the lighting
	
	result.a = diffuse.a;

	if(result.a < 0.05f)
		discard;


	return result;
//	return float4(1.0f,0.0f,0.0f,1.0f);
}


//--------------------------------------------------------------------------------------
// Techniques
//--------------------------------------------------------------------------------------
technique10 PosCol
{
    pass P0
    {   

//		SetBlendState( NoBlending, float4( 0.0f, 0.0f, 0.0f, 0.0f ), 0xFFFFFFFF );
//      SetDepthStencilState( DisableDepth, 0 );
 
        SetVertexShader( CompileShader( vs_4_0, VS_PosCol() ) );
        SetGeometryShader( NULL );
        SetPixelShader( CompileShader( ps_4_0, PS_LineShader() ) );
       
   }
}


technique10 PosTex
{
    pass P0
    {   

		SetDepthStencilState( DisableDepth, 0 );
		SetBlendState( TextBlending, float4( 0.0f, 0.0f, 0.0f, 0.0f ), 0xF);     
		SetRasterizerState(DataCulling);
        SetVertexShader( CompileShader( vs_4_0, VS_PosTex0() ) );
        SetGeometryShader( NULL );
        SetPixelShader( CompileShader( ps_4_0, PS_MeshShader(false,false) ) );
       
   }
}

technique10 PosColTex
{
	pass P0
    {   
		SetDepthStencilState( EnableDepthLessEqual, 0 );
		SetBlendState( Blending, float4( 1.0f, 1.0f, 1.0f, 1.0f ), 0xFFFFFFFF );
		SetRasterizerState(CullingCW);
        SetVertexShader( CompileShader( vs_4_0, VS_PosColTex0() ) );
        SetGeometryShader( NULL );
        SetPixelShader( CompileShader( ps_4_0, PS_MeshShader(true,true) ) );
   }
}

technique10 PosNorm
{
	pass P0
	{
//		SetDepthStencilState( EnableDepth, 0 );	
//		SetBlendState( NoBlending, float4( 0.0f, 0.0f, 0.0f, 0.0f ), 0xFFFFFFFF );
//		SetRasterizerState(DataCulling); 
        SetVertexShader( CompileShader( vs_4_0, VS_PosNorm() ) );
        SetGeometryShader( NULL );
        SetPixelShader( CompileShader( ps_4_0, PS_MeshShader(false,false) ) );	
	}
}

technique10 PosNormCol
{
	pass P0
	{
//		SetDepthStencilState( EnableDepth, 0 );	
        SetVertexShader( CompileShader( vs_4_0, VS_PosNormCol() ) );
        SetGeometryShader( NULL );
        SetPixelShader( CompileShader( ps_4_0, PS_MeshShader(false,true) ) );	
	}
}

technique10 PosNormTex
{
	pass P0
	{
	    SetVertexShader( CompileShader( vs_4_0, VS_PosNormTex0() ) );
        SetGeometryShader( NULL );
        SetPixelShader( CompileShader( ps_4_0, PS_MeshShader(false,false) ) );	
	}

}

// used for the cached meshes.  Even though tex isn't used it is stored in the mesh to prevent rebuilding
// when state changes.
technique10 PosNormColTex
{
	pass P0
	{
	    SetVertexShader( CompileShader( vs_4_0, VS_PosNormColTex0() ) );
        SetGeometryShader( NULL );
        SetPixelShader( CompileShader( ps_4_0, PS_MeshShader(false,true) ) );	
	}

}

technique10 Marker2D
{
    pass P0
    {   

//        SetDepthStencilState( DisableDepth, 0 );
        SetVertexShader( CompileShader( vs_4_0, VS_PosColTex0() ) );
        SetGeometryShader( NULL );
        SetPixelShader( CompileShader( ps_4_0, PS_MeshShader(true,true) ) );
       
   }
}

technique10 Marker3D
{
    pass P0
    {   

 //       SetDepthStencilState( Depth, 0 );
		SetRasterizerState(MarkerCulling);
        SetVertexShader( CompileShader( vs_4_0, VSG()));
        SetGeometryShader( CompileShader(gs_4_0, GSPoints()));
        SetPixelShader( CompileShader( ps_4_0, PS_LineShader()));
       
   }
}